const express = require('express');
const bcrypt = require('bcryptjs');
const jwt = require('jsonwebtoken');
// const path = require("path");
// const { ctrlWrapper, RequestError, sendMail, sendRequestRegistration } = require('../../helpers');
const { ctrlWrapper, RequestError, sendMail, paypal } = require('../../helpers');
const { validateBody, auth, passport } = require('../../middlewares');
const { User, schemas } = require('../../models/user');
const { Texture } = require('../../models/texture');
const { Project } = require('../../models/project');
// const { func } = require("joi");
const { v4: uuidv4 } = require('uuid');
// const { createCatalogTextures } = require('../../helpers/createCatalogTextures');

require('dotenv').config();

const router = express.Router();
const { ACCESS_TOKEN_SECRET_KEY, REFRESH_TOKEN_SECRET_KEY, REDIRECT_URL, VERIFY_URL, RESET_URL } =
    process.env;

router.post('/register', validateBody(schemas.signup), ctrlWrapper(register));
router.post('/login', validateBody(schemas.signin), ctrlWrapper(login));
// router.get('/confirm/:id', ctrlWrapper(confirm));
// router.get('/cancel/:id', ctrlWrapper(cancel));
router.get('/verify/:token', ctrlWrapper(verify));
router.post('/resend', validateBody(schemas.resend), ctrlWrapper(resend));
router.post('/message', auth, ctrlWrapper(sendMessage));
router.get('/logout', auth, ctrlWrapper(logout));
router.post('/refresh', validateBody(schemas.refresh), ctrlWrapper(refresh));
router.patch('/reset', validateBody(schemas.resend), ctrlWrapper(reset));
router.patch('/updateFavorite', auth, validateBody(schemas.favorite), ctrlWrapper(updateFavorite));
router.patch('/update', auth, validateBody(schemas.update), ctrlWrapper(update));
router.patch('/password/:passKey', validateBody(schemas.reset), ctrlWrapper(setPassword));
router.delete('/delete', auth, ctrlWrapper(deleteUser));
router.patch('/plan', auth, validateBody(schemas.pricingPlan), ctrlWrapper(setPricingPlan));
router.patch('/role', ctrlWrapper(setRole));
router.get('/all', ctrlWrapper(getAll));
router.get('/unsubscribe', auth, ctrlWrapper(unsubscribe));

router.get('/google', passport.authenticate('google', { scope: ['email', 'profile'] }));

router.get(
    '/google/callback',
    passport.authenticate('google', { session: false }),
    ctrlWrapper(socialAuth)
);

// router.get(
//     "/facebook",
//     passport.authenticate("facebook", { scope: ["public_profile", "email"] })
// );

// router.get(
//     "/facebook/callback",
//     passport.authenticate("facebook", { session: false }),
//     ctrlWrapper(socialAuth)
// );

router.get('/test', ctrlWrapper(sendEmailTest));

async function sendEmailTest(req, res) {
    // const mail = {
    //     from: 'rendersee.com',
    //     to: 'hellreebok@gmail.com',
    //     subject: 'Happyuser verification email',
    //     html: `<a target="_blank" href="${VERIFY_URL + 123}">Click to verify your email</a>`,
    // };

    const mail = {
        from: 'Rendersee.com <support@rendersee.com>',
        to: 'natanya1993@gmail.com',
        subject: 'Your verification email',
        html: `<div style="color:#020202; padding: 20px; text-align: center;">
        <h2>Rendersee</h2>
        <p style="font-size: 36px;">Verify your email address</p>
        </br>
        <p style="font-size: 20px; margin-bottom: 20px">To start using Rendersee.com just click the verify email button below:</p>
        </br>
        <a target="_blank" href="" style="text-align: center; font-weight: 700; text-decoration: none; padding: 10px 25px; border: 2px solid #020202; border-radius: 5px">Verify email</a>
        
        </div>`,
    };

    await sendMail(mail);

    res.json({ message: 'Email resend' });
}

async function register(req, res) {
    const { name, lastName, email, password } = req.body;
    const user = await User.findOne({ email });

    if (user && user.verify) {
        throw RequestError(409, 'Email in use');
    }

    if (user && !user.verify) {
        return res.status(201).json({
            user: { name: user.name, lastName: user.lastName, email: user.email },
        });
    }

    const hashPassword = await bcrypt.hash(password, 10);

    const token = uuidv4();

    const nextPayDay = new Date();
    nextPayDay.setDate(nextPayDay.getDate() + 14);

    const result = await User.create({
        name,
        lastName,
        email,
        password: hashPassword,
        verificationToken: token,
        nextPayDay,
    });

    const mail = {
        from: 'Rendersee.com <support@rendersee.com>',
        to: email,
        subject: 'Your verification email',
        html: `<div style="color:#020202; padding: 20px; text-align: center;">
            <h2>Rendersee</h2>
            <p style="font-size: 36px;">Verify your email address</p>
            </br>
            <p style="font-size: 20px; margin-bottom: 20px">To start using Rendersee.com just click the verify email button below:</p>
            </br>
            <a target="_blank" href="${
                VERIFY_URL + token
            }" style="text-align: center; font-weight: 700; text-decoration: none; padding: 10px 25px; border: 2px solid #020202; border-radius: 5px">Verify email</a>
            
            </div>`,
    };

    await sendMail(mail);

    res.status(201).json({
        user: {
            name: result.name,
            lastName: result.lastName,
            email: result.email,
            pricingPlan: result.pricingPlan,
            favorite: result.favorite,
            nextPayDay: result.nextPayDay,
            subscriptionId: result.subscriptionId,
            role: result.role,
        },
    });
}

async function verify(req, res) {
    // const { token } = req.body;
    const { token } = req.params;
    const user = await User.findOne({ verificationToken: token });
    if (!user) {
        throw RequestError(404, 'User not found');
    }
    await User.findByIdAndUpdate(user._id, {
        emailVerified: true,
        verify: true,
        verificationToken: null,
    });

    const { accessToken, refreshToken } = await updateTokens(user._id);
    const { name, lastName, pricingPlan, verify, favorite, nextPayDay, subscriptionId, role } =
        user;
    // await sendRequestRegistration(user);

    // res.status(201).json({
    //     accessToken,
    //     refreshToken,
    //     user: { name, lastName, email, pricingPlan, verify, favorite },
    // });
    res.redirect(
        `${REDIRECT_URL}?accessToken=${accessToken}&refreshToken=${refreshToken}&user=${JSON.stringify(
            { name, lastName, pricingPlan, verify, favorite, nextPayDay, subscriptionId, role }
        )}`
    );
}

// async function confirm(req, res) {
//     const { id } = req.params;
//     const user = await User.findByIdAndUpdate(id, {
//         verify: true,
//     });

//     const mail = {
//         to: user.email,
//         subject: 'Your registration has been confirmed',
//         html: `<h2> Your registration has been confirmed, please re-login.</h2>`,
//     };

//     await sendMail(mail);

//     res.send('<h1>User registration confirmed</h1>');
// }

// async function cancel(req, res) {
//     const { id } = req.params;
//     const user = await User.findByIdAndUpdate(id, {
//         banned: true,
//     });

//     const mail = {
//         to: user.email,
//         subject: 'Your registration has been rejected',
//         html: `<h2>Sorry, your registration was rejected.</h2>`,
//     };

//     await sendMail(mail);

//     res.send('<h1>User has been banned</h1>');
// }

async function resend(req, res) {
    const { email } = req.body;
    const user = await User.findOne({ email });
    if (!user) throw RequestError(401, 'Email not found');
    if (user.verify) throw RequestError(400, 'Email already verified');

    const mail = {
        to: email,
        subject: 'Happyuser verification email',
        html: `<a target="_blank" href="${
            VERIFY_URL + user.verificationToken
        }">Click to verify your email</a>`,
    };

    await sendMail(mail);

    res.json({ message: 'Email resend' });
}

async function reset(req, res) {
    const { email } = req.body;
    const passKey = uuidv4();
    const user = await User.findOneAndUpdate({ email }, { passKey });
    if (!user) throw RequestError(401, 'Email not found');

    const mail = {
        to: email,
        subject: 'Happyuser reset password',
        html: `<a target="_blank" href="${RESET_URL + passKey}">Click to reset your password</a>`,
    };

    await sendMail(mail);

    res.json({ message: 'Email sent' });
}

async function setPassword(req, res) {
    const { email, password } = req.body;
    const { passKey } = req.params;
    const user = await User.findOne({ email });
    if (!user) throw RequestError(401, 'Email not found');
    if (user.passKey !== passKey) throw RequestError(404, 'Wrong verification key');
    const hashPassword = await bcrypt.hash(password, 10);
    await User.findOneAndUpdate({ email }, { passKey: null, password: hashPassword });

    res.json({ message: 'Password changed' });
}

async function login(req, res) {
    const { email, password } = req.body;
    const user = await User.findOne({ email });

    if (!user) throw RequestError(401, 'Email or password is wrong');

    const passCompare = await bcrypt.compare(password, user.password);
    if (!passCompare) throw RequestError(401, 'Email or password is wrong');
    if (!user.emailVerified) throw RequestError(401, 'Email not verified');
    if (user.banned) throw RequestError(401, 'User has been banned');

    const { accessToken, refreshToken } = await updateTokens(user._id);
    const { name, lastName, pricingPlan, verify, favorite, nextPayDay, subscriptionId, role } =
        user;

    res.status(200).json({
        accessToken,
        refreshToken,
        user: {
            name,
            lastName,
            pricingPlan,
            email,
            verify,
            favorite,
            nextPayDay,
            subscriptionId,
            role,
        },
    });
}

async function logout(req, res) {
    const { _id } = req.user;

    await User.findByIdAndUpdate(_id, {
        accessToken: null,
        refreshToken: null,
    });

    res.status(204).json();
}

async function refresh(req, res) {
    try {
        const { refreshToken: token } = req.body;
        const { id } = jwt.verify(token, REFRESH_TOKEN_SECRET_KEY);

        const user = await User.findById(id);
        if (!user || !user.refreshToken) throw RequestError(401, 'Unautharized');

        const { accessToken, refreshToken } = await updateTokens(user._id, user.subscribeToken);
        let { name, lastName, email, pricingPlan, favorite, nextPayDay, subscriptionId, role } =
            user;

        const today = new Date();
        if (today > nextPayDay) {
            pricingPlan = 'None';
            await User.findByIdAndUpdate(id, { pricingPlan });
        }

        res.status(200).json({
            accessToken,
            refreshToken,
            user: {
                name,
                lastName,
                email,
                pricingPlan,
                favorite,
                nextPayDay,
                subscriptionId,
                role,
            },
        });
    } catch (error) {
        if (!error.status) error.status = 401;
        throw error;
    }
}

async function socialAuth(req, res) {
    const { user } = req;

    const { accessToken, refreshToken } = await updateTokens(user._id);
    // await createCatalogTextures(user._id);

    const {
        name,
        lastName,
        email,
        pricingPlan,
        verify,
        favorite,
        nextPayDay,
        subscriptionId,
        role,
    } = user;
    // await sendRequestRegistration(user);
    const userToSend = {
        name,
        lastName,
        email,
        pricingPlan,
        verify,
        favorite,
        nextPayDay,
        subscriptionId,
        role,
    };
    res.redirect(
        `${REDIRECT_URL}?accessToken=${accessToken}&refreshToken=${refreshToken}&user=${JSON.stringify(
            userToSend
        )}`
    );
}

async function updateTokens(id, subscribeToken) {
    const accessToken = jwt.sign({ id }, ACCESS_TOKEN_SECRET_KEY, {
        expiresIn: '1h',
    });
    const refreshToken = jwt.sign({ id }, REFRESH_TOKEN_SECRET_KEY, {
        expiresIn: '30d',
    });

    // try {
    //     const { plan} = jwt.verify(token, REFRESH_TOKEN_SECRET_KEY);
    // }

    await User.findByIdAndUpdate(id, {
        accessToken,
        refreshToken,
    });

    return { accessToken, refreshToken };
}

// async function getSubscribeTokens(plan) {
//     const subscribeToken = jwt.sign({ plan }, SUBSCRIBE_TOKEN_SECRET_KEY, {
//         expiresIn: '30d',
//     });

//     // await User.findByIdAndUpdate(id, { subscribeToken });

//     return { accessToken, refreshToken };
// }

async function deleteUser(req, res) {
    const { _id } = req.user;

    const user = await User.findById(_id);

    if (!user) throw RequestError(404, 'Not found');
    if (user.subscriptionId) {
        paypal.cancelSubscription(user.subscriptionId); //without await
    }
    await User.findByIdAndDelete(_id);

    res.sendStatus(200);
}

async function setPricingPlan(req, res) {
    const user = req.user;
    const { subscriptionId, plan_id } = req.body;

    // const plan = {
    //     basic: 'P-07R859561S031722VMUAWA5Y',
    //     basic14: 'P-9Y493700K0857335UMUADK5Y', //has free trial
    //     // enterprice: 'P-3XS4801330882222PMUAEW3I',
    //     enterprice: 'P-5RN91923AL8155529MUEVDQQ',
    // }; //sandbox

    const plan = {
        basic: 'P-1KD769117E528273XMUFNYDI',
        // basic14: 'P-8WM48351LR382992RMT3V7CA', //has free trial,
        enterprice: 'P-5GN17812N3892143KMT3WAFI',
    };

    const pricingPlan = plan_id === plan.enterprice ? 'Enterprice' : 'Basic';
    // const hasTrial = plan_id === plan.enterprice ? user.hasTrial : false;
    const nextPayDay = new Date();
    // nextPayDay.setHours(0, 0, 0, 0);
    // const period = plan_id === plan.enterprice ? 1 : user.hasTrial ? 15 : 1;

    nextPayDay.setDate(nextPayDay.getDate() + 1);

    const result = await User.findByIdAndUpdate(
        user._id,
        { pricingPlan, subscriptionId, nextPayDay },
        { new: true }
    );
    if (!result) throw RequestError(404, 'Not found');

    if (user.subscriptionId) {
        paypal.cancelSubscription(user.subscriptionId); //without await
    }
    res.status(200).json(result.pricingPlan);
}

async function updateFavorite(req, res) {
    const { _id } = req.user;
    const { id } = req.body;
    const user = await User.findById(_id);
    if (!user) throw RequestError(404, 'Not found');
    const favorite = [...user.favorite];
    const index = favorite.indexOf(id);

    if (index < 0) favorite.push(id);
    else favorite.splice(index, 1);
    await User.findByIdAndUpdate(_id, { favorite });

    res.sendStatus(200);
}

async function sendMessage(req, res) {
    const user = req.user;

    const { firstName, lastName, email, tel, message } = req.body;

    const mail = {
        to: 'hellreebok@gmail.com, support@rendersee.com',
        subject: `${user.name} ${user.lastName} left a message on the rendersee.com`,

        text: `
        \r\n
        First name: ${user.name}\r\n
        Last name: ${user.lastName}\r\n
        Email: ${user.email}\r\n
        \r\n
        Message: ${message}`,
    };

    await sendMail(mail);

    res.json({ message: 'Message sent' });
}

async function getAll(req, res) {
    // const { page = 1, limit = 20, favorite = [true, false] } = req.query;
    const users = await User.find();
    const textures = await Texture.find();
    const projects = await Project.find();

    const result = users.map(user => {
        const owner = user._id.toString();
        // console.log('owner: ', owner);
        const numberOfTextures = textures.filter(
            texture => texture.owner.toString() === owner
        ).length;
        const numberOfProjects = projects.filter(
            project => project.owner.toString() === owner
        ).length;

        return { user, numberOfTextures, numberOfProjects };
    });

    res.status(200).json(result);
}

async function setRole(req, res) {
    const { id, role } = req.body;
    const user = await User.findByIdAndUpdate(id, { role });

    if (!user) throw RequestError(404, 'Not found');

    res.sendStatus(200);
}

async function unsubscribe(req, res) {
    const { _id } = req.user;
    const user = await User.findById(_id);

    if (!user) throw RequestError(404, 'Not found');
    paypal.cancelSubscription(user.subscriptionId); //without await
    res.sendStatus(200);
}

async function update(req, res) {
    const { _id } = req.user;
    const user = await User.findByIdAndUpdate(_id, req.body, {
        new: true,
    });

    res.status(201).json(req.body);
}

module.exports = router;
