const express = require('express');
const path = require('path');
// const fs = require("fs/promises");
const fs = require('fs').promises;
const { v4: uuidv4 } = require('uuid');
const { Texture, schemas } = require('../../models/texture');
const { RequestError, ctrlWrapper } = require('../../helpers');
const { validateBody, auth, upload } = require('../../middlewares');

const router = express.Router();

const texturesDir = path.join(__dirname, '../../', 'public', 'textures');
// const texturesDir = path.join(__dirname, '../../', 'public', 'textures');

router.get('/', auth, ctrlWrapper(getAll));
router.post('/', auth, upload.single('imgFile'), validateBody(schemas.add), ctrlWrapper(add));
router.get('/:id', auth, ctrlWrapper(getById));
router.delete('/:id', auth, ctrlWrapper(remove));
router.patch(
    '/favorite/:id',
    auth,
    validateBody(schemas.favorite),
    ctrlWrapper(updateFavoriteStatus)
);
router.patch(
    '/update/:id',
    auth,
    upload.single('imgFile'),
    validateBody(schemas.add),
    ctrlWrapper(update)
);

async function getAll(req, res) {
    const { _id: owner } = req.user;
    // const { page = 1, limit = 20, favorite = [true, false] } = req.query;
    const result = await Texture.find({ owner }, '-createdAt -updatedAt').populate('owner', 'name');

    res.status(200).json(result);
}

async function add(req, res) {
    const { name, width, height, type } = req.body;
    const { _id: owner } = req.user;
    const { path: tempUpload, originalname } = req.file;

    const extension = originalname.split('.').pop();
    const imgName = originalname.split('.').shift() + uuidv4();

    const filename = `${imgName}.${extension}`;
    const resultUpload = path.join(texturesDir, filename);
    await fs.rename(tempUpload, resultUpload);

    const img = path.join('textures', filename);

    const result = await Texture.create({
        name,
        type,
        width,
        height,
        img,
        // origin: 'uploads',
        owner,
    });

    res.status(201).json(result);
}

async function getById(req, res) {
    const { id } = req.params;
    const result = await Texture.findById(id);
    if (!result) throw RequestError(404, 'Not found');

    res.status(200).json(result);
}

async function remove(req, res) {
    const { id } = req.params;
    const result = await Texture.findByIdAndRemove(id);
    if (!result) throw RequestError(404, 'Not found');

    res.status(204).json({ message: 'texture deleted' });
}

async function updateFavoriteStatus(req, res) {
    const { id } = req.params;
    const isTexture = await Texture.findById(id);
    if (!isTexture) throw RequestError(404, 'Texture not found');
    const result = await Texture.findByIdAndUpdate(id, req.body, {
        new: true,
    });

    res.status(200).json(result);
}

async function update(req, res) {
    const { id } = req.params;
    const isTexture = await Texture.findById(id);
    if (!isTexture) throw RequestError(404, 'Texture not found');

    // console.log('req.body: ', req.body);
    if (req.file) {
        const { path: tempUpload } = req.file;
        const fileName = isTexture.img.split('/')[1];
        const resultUpload = path.join(texturesDir, fileName);
        await fs.rename(tempUpload, resultUpload);
    }
    const result = await Texture.findByIdAndUpdate(id, req.body, {
        new: true,
    });

    res.status(200).json(result);
}

module.exports = router;
