const express = require('express');
const path = require('path');
const fs = require('fs').promises;
const { v4: uuidv4 } = require('uuid');
const { Project, schemas } = require('../../models/project');
const { RequestError, ctrlWrapper } = require('../../helpers');
const { validateBody, auth, upload } = require('../../middlewares');

const router = express.Router();

const screenshotDir = path.join(__dirname, '../../', 'public', 'screenshots');

router.get('/', auth, ctrlWrapper(getAll));
router.post('/', auth, upload.single('img'), validateBody(schemas.add), ctrlWrapper(add));
router.patch(
    '/update/:id',
    auth,
    upload.single('img'),
    validateBody(schemas.update),
    ctrlWrapper(update)
);
router.delete('/:id', auth, ctrlWrapper(remove));

async function getAll(req, res) {
    const { _id: owner } = req.user;
    const result = await Project.find({ owner }).populate('owner', 'name');

    res.status(201).json(result);
}

async function add(req, res) {
    const { name, room, meshes } = req.body;
    const meshesObj = JSON.parse(meshes);
    const { _id: owner } = req.user;

    const { path: tempUpload, originalname } = req.file;
    const extension = originalname.split('.').pop();
    const imgName = originalname.split('.').shift() + uuidv4();

    const filename = `${imgName}.${extension}`;
    const resultUpload = path.join(screenshotDir, filename);
    await fs.rename(tempUpload, resultUpload);

    const img = path.join('screenshots', filename);
    // const img = '';

    const result = await Project.create({ name, room, img, meshes: meshesObj, owner });

    res.status(201).json(result);
}

async function remove(req, res) {
    const { id } = req.params;
    const result = await Project.findByIdAndRemove(id);
    if (!result) throw RequestError(404, 'Not found');

    res.status(204).json({ message: 'Project deleted' });
}

async function update(req, res) {
    const { id } = req.params;
    const { name, room, meshes } = req.body;
    let meshesObj, fields;
    if (meshes) {
        meshesObj = JSON.parse(meshes);
        fields = { name, room, meshes: meshesObj };
    } else fields = { name };
    // console.log('fields: ', fields);

    if (req.file) {
        const { path: tempUpload, originalname } = req.file;
        const extension = originalname.split('.').pop();
        const imgName = originalname.split('.').shift() + uuidv4();

        const filename = `${imgName}.${extension}`;
        const resultUpload = path.join(screenshotDir, filename);
        await fs.rename(tempUpload, resultUpload);

        const img = path.join('screenshots', filename);
        fields.img = img;
    }

    const result = await Project.findByIdAndUpdate(id, fields, { new: true });
    if (!result) throw RequestError(404, 'Not found');

    res.status(200).json(result);
}

module.exports = router;
